#include "_Uart.h"
#include "GPIORegs.h"
#include "INTRRegs.h"
#include "INTRDefs.h"
#include <stdarg.h>

extern void HaltFunc();

#define W_REG(reg, data) (*reg) = (data)
#define R_REG(reg) (*reg)

static void UartIntr (void) __attribute__ ((interrupt ("IRQ")));

/*
 * UART Initialization
 */
void _Uart_Init(U32 u32Pclk, U32 u32baud, U32 u32WhichTxUART)
{
	U32 u32Offset = UOFFSET * (u32WhichTxUART);
	U32 u32Div = 0;
	
    /*
        FIFO control register, FIFO disable
    */
    
    
	*(UFCON0 + u32Offset) = UFCON_RESET;

    /*
        MODEM control register, AFC disable
    */
	*(UMCON0 + u32Offset) = UMCON_RESET;

    /*
        Line control register : Normal,No parity,1 stop,8 bits
    */
	*(ULCON0 + u32Offset) = (ULCON_MODE_NORM | ULCON_PARITY_NONE | ULCON_STOP_ONE | ULCON_LENGTH_8);

    /*
        Uart Control register
    */
	*(UCON0 + u32Offset) = (UCON_CLK_PCLK | UCON_TXINT_LEVEL | UCON_RXERR_ENABLE | UCON_TXMODE_INT | UCON_RXMODE_INT);

	switch (u32baud)
	{
		case 115200:
			switch (u32Pclk)
			{
				case 50000000:
					u32Div = 26;
					break;
				default:
					while(1);
			}
			break;
		default:
			HaltFunc ();		
			while (1);
	}

    /*
        Baud rate divisior register
    */
	*(UBRDIV0 + u32Offset) = u32Div;

	__asm("nop");
	__asm("nop");
	__asm("nop");
	__asm("nop");
	__asm("nop");
	__asm("nop");
	__asm("nop");
	__asm("nop");
	__asm("nop");
	__asm("nop");
}

/*
 * ==============================================================================================================
 */

void _Uart_SendChar(U32 u32WhichTxUART, char cChar)
{
	U32 u32Offset = UOFFSET * (u32WhichTxUART);
	switch((*(UFCON0 + u32Offset)) & UFCON_FIFO)
	{
		case UFCON_FIFO_ENABLE:
			while((*(UFSTAT0 + u32Offset)) & UFSTAT_TXFULL);
			break;
		case UFCON_FIFO_DISABLE:
			while(!((*(UTRSTAT0 + u32Offset)) & UTRSTAT_TX));   //Wait until THR is empty.
			break;
		default:
			break;

	}

	W_REG((UTXH0 + u32Offset), cChar);
}

/*
 * ==============================================================================================================
 */
void _Uart_SendString(U32 u32WhichTxUART, char *sString)
{
	while(*sString)
	{
		_Uart_SendChar(u32WhichTxUART, *(sString++));
	}
}



/*
 * ==============================================================================================================
 */
char _Uart_Getch(unsigned long u32WhichUart)
{
	U32 u32Offset = UOFFSET * (u32WhichUart);
	switch((*(UFCON0 + u32Offset)) & UFCON_FIFO)
	{
		case UFCON_FIFO_ENABLE:
			while(!((*(UFSTAT0 + u32Offset)) & UFSTAT_RXCOUNT));
			break;
		case UFCON_FIFO_DISABLE:
			while(!((*(UTRSTAT0 + u32Offset)) & UTRSTAT_RX_BUF));   //Wait until THR is empty.
			break;
		default:
			break;
	}
	return (char)(*(URXH0 + u32Offset));
}

char _Uart_Getkey(unsigned long u32WhichUart)
{
	U32 u32Offset = UOFFSET * (u32WhichUart);
	switch((*(UFCON0 + u32Offset)) & UFCON_FIFO)
	{
		case UFCON_FIFO_ENABLE:
			if (((*(UFSTAT0 + u32Offset)) & UFSTAT_RXCOUNT))
				return (char)(*(URXH0 + u32Offset));
			break;
		case UFCON_FIFO_DISABLE:
			if (((*(UTRSTAT0 + u32Offset)) & UTRSTAT_RX_BUF))
				return (char)(*(URXH0 + u32Offset));			
			break;
		default:
			break;
	}
	return 0;
}



void TestUartLoopbackNoFifo ()
{
	U32 u32WhichUART;
	char cByte;
	char cTmp;
	
	*INTMSK |=  (BIT_UART1 | BIT_UART2);	
		
	Uart_Printf ("\nPlease enter the selected UART channels [1 or 2]...");
	u32WhichUART = Uart_GetIntNum ();
	
	if ((u32WhichUART != 1) && (u32WhichUART != 2))
	{
		Uart_Printf ("Error: Bad channel selection");	
		return;
	}
	
	Uart_Printf ("\nPlease enter the byte to be transmitted through Loopback channel...");	
	cByte = Uart_Getch ();
	
	/*
	 * Now- set up the selected Uart. Please pay attention - you must not use hardware control flow.
	 * Do not forget GPIO settings.
	 * Please set to No-FIFO mode. 
	 * Please set 115200 bps - does it really matter?
	 * Please use UART CONTROL REGISTER to set loopback mode if transmitter and receiver are the smae UART. 
	*/
  	//*GPHCON |= (((unsigned long)(0xA)) << (8 + ((u32WhichUART - 1)<<2)));
	//*GPHUP |=  (((unsigned long)0x3)<< (2 + ((u32WhichUART - 1)<<1)));

	/* Enable polling mode for transmit and receive */	
	*(UCON0 + (0x1000 * (u32WhichUART))) = 0x5;	

	/* Flush FIFOs and set to non-FIFO mode */	
	*(UFCON0 + (0x1000 * (u32WhichUART))) = 0x6;
	
	
	/* Now issue a delay to let reset complete*/
	Timer_Bdelay_Milli (1);
	
	/*
		Set frame to Normal,No parity,1 stop,8 bits
		Use ULCON registers
	*/
    *(ULCON0 + (0x1000 * (u32WhichUART))) = 0x3;	

	/* Set loopback */			
	*(UCON0 + (0x1000 * (u32WhichUART))) |= 0x20;

	/* Set 115200 */	
	*(UBRDIV0 + (0x1000 * (u32WhichUART))) = 26;			
	
	
	/* Now - send the byte. */
	while(!((*(UTRSTAT0 + (0x1000 * u32WhichUART))) & UTRSTAT_TX));   //Wait until THR is empty.
	*(UTXH0_BYTE + (0x4000 * (u32WhichUART))) = cByte;		
	
	/* Now - read the byte and verify it is the one you sent */
	while(!((*(UTRSTAT0 + (0x1000 * u32WhichUART))) & UTRSTAT_RX_BUF));   //Wait until data is valid.
	cTmp = *(URXH0_BYTE + (0x4000 * u32WhichUART));
	
	if (cTmp == cByte)
	{
		Uart_Printf ("Test completed successfuly\n");	
	}
	else
	{
		Uart_Printf ("Expected 0x%x vs Received 0x%x\n", cByte, cTmp);		
	} 
	
}

void TestUartLoopbackNoFifoGenerateError ()
{
	U32 u32WhichTxUART;
	U32 u32WhichRxUART;	
	volatile char cByte;
	U32 error = 1;
		
	Uart_Printf ("\nPlease enter the selected TX UART channel [1 or 2]...");
	u32WhichTxUART = Uart_GetIntNum ();
	
	if ((u32WhichTxUART != 1) && (u32WhichTxUART != 2))
	{
		Uart_Printf ("Error: Bad channel selection");	
		return;
	}

	Uart_Printf ("\nPlease enter the selected RX UART channel [1 or 2]...");
	u32WhichRxUART = Uart_GetIntNum ();
	
	if ((u32WhichRxUART != 1) && (u32WhichRxUART != 2))
	{
		Uart_Printf ("Error: Bad channel selection");	
		return;
	}
	
	if (u32WhichRxUART == u32WhichTxUART)
	{
		Uart_Printf ("Error: Bad channel selection");	
		return;
	}	
	
	
	Uart_Printf ("\nPlease enter the byte to be transmitted ...");	
	cByte = Uart_Getch ();
	
	/*
	 * Now- set up the selected Uart. Please pay attention - you must not use hardware control flow.
	 * Do not forget GPIO settings.
	 * Please set to No-FIFO mode. 
	 * Please set 115200 bps?
	 * Please use UART CONTROL REGISTER to set loopback mode if transmitter and receiver are the smae UART.
	*/
  	*GPHCON |= (((unsigned long)(0xAA)) << 8);
	*GPHUP |=  0x3C; 	
	
	
	/* Now - generate the error. */

	
	/* Now - verify that you got an error */
	/* Use UART ERROR STATUS REGISTER from page 348 */
}


/*
 * ==============================================================================================================
 */
#define	ESC_KEY		0x1b
volatile int	iStringLength = 0;
volatile int    iCollectedBytes = 0;
char            cLoopBackRecvBuffer[128];
char	        cString[65];

static void  UartIntr(void)
{
	volatile U32 rUart;


    if (*INTPND & BIT_UART1) {
    	rUart = *UTRSTAT1;
    	if(rUart & 0x2) {   
			*SUBSRCPND = BIT_SUB_TXD1;    		
        }	
    	if(rUart & 0x1) {   
			while ((*UFSTAT1) & 0x3F)
			{
				/*
					Receive data ready
				*/
				cLoopBackRecvBuffer[iCollectedBytes ++] = *(URXH0_BYTE + (0x4000 * (1)));
				iStringLength --;
			}

			cLoopBackRecvBuffer[iCollectedBytes ++] = ' ';
			*SUBSRCPND = BIT_SUB_RXD1;
        }
		*SRCPND = BIT_UART1;
		*INTPND = BIT_UART1;
    }

    if (*INTPND & BIT_UART2) {
    	rUart = *UTRSTAT2;
    	if(rUart & 0x2) {
			*SUBSRCPND = BIT_SUB_TXD2;    		
        }	
    	if(rUart & 0x1) {   
			while (*(UFSTAT2) & 0x3F)
			{
				/*
					Receive data ready
				*/
				cLoopBackRecvBuffer[iCollectedBytes ++] = *(URXH0_BYTE + (0x4000 * (2)));
				iStringLength --;
			}
			
			cLoopBackRecvBuffer[iCollectedBytes ++] = ' ';
			*SUBSRCPND = BIT_SUB_RXD2;
			
        }
		*SRCPND = BIT_UART2;
		*INTPND = BIT_UART2;
    }
}


void Test_UART_LoopbackFIFOIntr ( void )
{
	U32     u32WhichTxUART;
	U32     u32WhichRxUART;	
	char *	pcString = (char *)cString;
	int     iKey = 0;


	
   	iStringLength = 0;
    iCollectedBytes = 0;
    
  	
    /*
     * TODO -
     *  Disable interrupts from UART1 and UART2
     *  Disable SubInterrupts from UART1 and UART2 
	*/
	*INTMSK |=  (BIT_UART1 | BIT_UART2);
	*INTSUBMSK  |= ((BIT_SUB_TXD1 | BIT_SUB_RXD1 | BIT_SUB_TXD2 | BIT_SUB_RXD2));

	
	/*
		In this test we will use the default baud rate (115200 bps) and frame 
		settings for the selected UARTs. However, the selected UARTs
		registers will be adjusted to allow FIFO mode. In case the reviever and 
		the trasmitter are the same, loopback bit is turned on.
		We duplicate initializations for both UARTs, though it's not neccessaary,
		since bits that are not significant to one UART are significant to the other.
	*/
	Uart_Printf ("\nPlease enter the selected Transmitting UART channel [1 or 2]...");
	u32WhichTxUART = Uart_GetIntNum ();


	if ((u32WhichTxUART != 1) && (u32WhichTxUART != 2))
	{
		Uart_Printf ("Error: Bad channel selection");	
		return;
	}

	
	
	Uart_Printf ("\nPlease enter the selected Recieveing UART channel [1 or 2]...");
	u32WhichRxUART = Uart_GetIntNum ();
	

	if ((u32WhichRxUART != 1) && (u32WhichRxUART != 2))
	{
		Uart_Printf ("Error: Bad channel selection");	
		return;
	}
	
	if (u32WhichRxUART != u32WhichRxUART)
	{	
	    /*
	     * Additional setting
	     * Enable GPIO 
	  	*/
	  	*GPHCON |= (((unsigned long)(0xAA)) << 8);
		*GPHUP |=  0x3C;
	} 	
	

	/*
	* TODO - Set iSRs for UART1 and UART2 
	*/
	
    pISR_UART1 = (U32)UartIntr;	
    pISR_UART2 = (U32)UartIntr;	    
    
    	
	Uart_Printf ("\nPlease enter a string (64 charcters MAX)...");	
	Uart_GetString (cString);

	/*
		The memory locations of the UARTs differ by 0x4000 bytes
		or 0x1000 words from each other.
		Therefore, we use the registers of UART0 as the base addresses
		and add a 0x4000 or 0x8000 bytes in order to access the right
		one.
	*/
	
	
	/*
		Set selected UARTs to provide pulse interrupts,
		and get recieve timeout interrupts.
		Use UCON registers
	*/
	*(UCON0 + (0x1000 * (u32WhichTxUART))) = 0xc5;
	if (u32WhichRxUART != u32WhichTxUART)
	{
		*(UCON0 + (0x1000 * (u32WhichRxUART))) = 0xc5;				
	}



	/*
		We will reset UART sFIFOs, in order to clear any redundant
		FIFO data.
		Use UFCON regsiters.
	*/
	*(UFCON0 + (0x1000 * (u32WhichTxUART))) = 0x6;		
	if (u32WhichRxUART != u32WhichTxUART)
	{
		*(UFCON0 + (0x1000 * (u32WhichRxUART))) = 0x6;				
	}

		
	/*
		Let's let the reset operation to complete.
	*/		
	Timer_Bdelay_Milli (1);
		
	/*
		FIFO mode enabled, reciever threshold is 8 bytes,
		transmitter threshold is 0 bytes.
		We will get a reciever interrupt either once we
		get at least 8 bytes, or a timeout occured (we have
		to set also the line control register for that.
		Use UFCON registers.
	*/
	*(UFCON0 + (0x1000 * (u32WhichTxUART))) = 0x11;	
	if (u32WhichRxUART != u32WhichTxUART)
	{
		*(UFCON0 + (0x1000 * (u32WhichRxUART))) = 0x11;				
	}
	

	
	/*
		Set frame to Normal,No parity,1 stop,8 bits
		Use ULCON registers
	*/
    *(ULCON0 + (0x1000 * (u32WhichTxUART))) = 0x3;			
	if (u32WhichRxUART != u32WhichTxUART)
	{
		*(ULCON0 + (0x1000 * (u32WhichRxUART))) = 0x3;				
	}
    


	/*
		If the test is imposed on one UART, then set the loopback enable bit
	*/
	if (u32WhichRxUART == u32WhichTxUART)
	{
		*(UCON0 + (0x1000 * (u32WhichTxUART))) |= 0x20;	
	}
	
	/*
		Set 115200 baud rate.
		Use UBRDIV registers.
	*/
	*(UBRDIV0 + (0x1000 * (u32WhichTxUART))) = 26;	
	if (u32WhichRxUART != u32WhichTxUART)
	{
		*(UBRDIV0 + (0x1000 * (u32WhichRxUART))) = 26;				
	}

	
	/*
		Since we use FIFO mode, there is no fear that we will overrun
		the transmit FIFO which is 64 bytes.

	*/
	while (*pcString)
	{
		/*
			rUTXH0 is a pointer to a byte address.
			Write one byte of the string to the UART transmitter FIFO	
		*/	
		*(UTXH0_BYTE + (0x4000 * (u32WhichTxUART))) = *pcString;
		pcString ++;
		iStringLength ++;
	}


	/*
	* Enable interrupts.
	*/
    *INTSUBMSK &= ~((BIT_SUB_TXD0 << (u32WhichTxUART * 3)) | (BIT_SUB_RXD0 << (u32WhichRxUART * 3)));
	*INTMSK &= ~ (BIT_UART1 | BIT_UART2);    
	
	
    while ((iStringLength != 0) && ((iKey = _Uart_Getkey(0)) != ESC_KEY));

	/*
	 * 	Now, disable interrupts.
	*/     
	
	*INTMSK |=  (BIT_UART1 | BIT_UART2);
	*INTSUBMSK  |= ((BIT_SUB_TXD1 | BIT_SUB_RXD1 | BIT_SUB_TXD2 | BIT_SUB_RXD2));
     

	/*
		If the test is imposed on one UART, then restore the from loopback mode
	*/
	
	if (u32WhichRxUART == u32WhichTxUART)
	{
		*(UCON0 + (0x1000 * (u32WhichTxUART))) &= ~0x20;	
	}

	
    if (iKey == ESC_KEY)
    	Uart_Printf ("Abnormal termination of Uart Loopback test\n");	
    else
    {	
		for (; iStringLength < iCollectedBytes;)
			Uart_Printf ("%c", cLoopBackRecvBuffer[iStringLength ++]); 
    	Uart_Printf ("\nUart Loopback test completed successfuly\n");				
	}
}
