#include "types.h"
#include "Utils.h"
#include "IICRegs.h"
#include "IICDefs.h"
#include "GPIORegs.h"
#include "INTRRegs.h"
#include "INTRDefs.h"

#define I2C_TIMEOUT 1

static void i2c_init ();

static int i2c_probe (unsigned char chip);

static int i2c_transfer (unsigned char cmd_type,
		  unsigned char chip,
		  unsigned char addr[],
		  unsigned char addr_len,
		  unsigned char data[], unsigned short data_len);
		  
static void ReadWriteByte (void);

static int WaitForXfer (void);		  

static int IsACK (void);

void Test_IIC_Probe (void)
{
	int j;
	
	i2c_init ();
	
	for (j = 0; j < 128; j++)
	{
		if (i2c_probe(j) == 0)
			Util_Printf(" Found on 0x%02X\n", j);
	}	
}

void i2c_init ()
{
    *GPEUP  |= 0xc000;                  //Pull-up disable
    *GPECON &= ~0xF0000000;             //GPE15:IICSDA , GPE14:IICSCL    
    *GPECON |= 0xA0000000;              //GPE15:IICSDA , GPE14:IICSCL
	*IICCON = 0xE0;					    // Enable acknowlege and pending data notfication, IIC clock = PCLK/512     
}


int i2c_probe (unsigned char chip)
{
	unsigned char buf[1];

	buf[0] = 0;

	/*
	 * What is needed is to send the chip address and verify that the
	 * address was <ACK>ed (i.e. there was a chip at that address which
	 * drove the data line low).
	 */
	return (i2c_transfer (I2C_READ, chip << 1, 0, 0, buf, 1) != I2C_OK);
}

/*
 *  ReadWriteByte must be called in one of the following states:
 * - after data has been written to *IICDS (data shift register)
 * - before waiting for incoming data that will be read later from *IICDS
 * - after a stop signal was sent through *IICSTAT.   
*/ 
void ReadWriteByte (void)
{
	*IICCON &= ~I2CCON_IRPND;
}

int WaitForXfer (void)
{
	int i, status;

	i = I2C_TIMEOUT * 10000;
	status = *IICCON;
	while ((i > 0) && !(status & I2CCON_IRPND)) {
		Timer_Bdelay_Micro (100);
		status = *IICCON;
		i--;
	}
	return (status & I2CCON_IRPND) ? I2C_OK : I2C_NOK_TOUT;
}

int IsACK (void)
{
	return (!(*IICSTAT & I2CSTAT_NACK));
}



/*
 * cmd_type is 0 for write, 1 for read.
 *
 * addr_len can take any value from 0-255, it is only limited
 * by the char, we could make it larger if needed. If it is
 * 0 we skip the address write cycle.
 */
int i2c_transfer (unsigned char cmd_type,
		  unsigned char chip,
		  unsigned char addr[],
		  unsigned char addr_len,
		  unsigned char data[], unsigned short data_len)
{
	int i, status, result;
	
	if (data == 0 || data_len == 0) {
		/*Don't support data transfer of no length or to address 0 */
		Util_Printf ("i2c_transfer: bad call\n");
		return I2C_NOK;
	}

	/* Check I2C bus idle */
	i = I2C_TIMEOUT * 1000;
	status = *IICSTAT;
	while ((i > 0) && (status & I2CSTAT_BSY)) {
		Timer_Bdelay_Micro (1000);
		status = *IICSTAT;
		i--;
	}
	

	if (status & I2CSTAT_BSY)
		return I2C_NOK_TOUT;
		
	result = I2C_OK;

	switch (cmd_type) {
	case I2C_WRITE:
		if (addr && addr_len) {
			*IICDS = chip;
			/* send START */
			*IICSTAT = I2C_MODE_MT | I2C_TXRX_ENA | I2C_START_STOP;
			i = 0;
			while ((i < addr_len) && (result == I2C_OK)) {
				result = WaitForXfer ();
				*IICDS = addr[i];
				ReadWriteByte ();
				i++;
			}
			i = 0;
			while ((i < data_len) && (result == I2C_OK)) {
				result = WaitForXfer ();
				*IICDS = data[i];
				ReadWriteByte ();
				i++;
			}
		} else {
			*IICDS = chip;
			/* send START */
			*IICSTAT = I2C_MODE_MT | I2C_TXRX_ENA | I2C_START_STOP;
			i = 0;
			while ((i < data_len) && (result = I2C_OK)) {
				result = WaitForXfer ();
				*IICDS = data[i];
				ReadWriteByte ();
				i++;
			}
		}

		if (result == I2C_OK)
			result = WaitForXfer ();

		/* send STOP */
		*IICSTAT = I2C_MODE_MR | I2C_TXRX_ENA;
		ReadWriteByte ();
		break;

	case I2C_READ:
		if (addr && addr_len) {
			*IICSTAT = I2C_MODE_MT | I2C_TXRX_ENA;
			*IICDS = chip;
			/* send START */
			*IICSTAT |= I2C_START_STOP;
			result = WaitForXfer ();
			if (IsACK ()) {
				i = 0;
				while ((i < addr_len) && (result == I2C_OK)) {
					*IICDS = addr[i];
					ReadWriteByte ();
					result = WaitForXfer ();
					i++;
				}

				*IICDS = chip;
				/* resend START */
				*IICSTAT =  I2C_MODE_MR | I2C_TXRX_ENA |
						I2C_START_STOP;
				ReadWriteByte ();
				result = WaitForXfer ();
				i = 0;
				while ((i < data_len) && (result == I2C_OK)) {
					/* disable ACK for final READ */
					if (i == data_len - 1)
						*IICCON &= ~0x80;
					ReadWriteByte ();
					result = WaitForXfer ();
					data[i] = *IICDS;
					i++;
				}
			} else {
				result = I2C_NACK;
			}

		} else {
			*IICSTAT = I2C_MODE_MR | I2C_TXRX_ENA;
			*IICDS = chip;
			/* send START */
			*IICSTAT |= I2C_START_STOP;
			result = WaitForXfer ();
			
			if (IsACK ()) {
				i = 0;
				while ((i < data_len) && (result == I2C_OK)) {
					/* disable ACK for final READ */
					if (i == data_len - 1)
						*IICCON &= ~0x80;
					ReadWriteByte ();
					result = WaitForXfer ();
					data[i] = *IICDS;
					i++;
				}
					
			} else {
				result = I2C_NACK;
			}
		}

		/* send STOP */
		*IICSTAT = I2C_MODE_MR | I2C_TXRX_ENA;
		ReadWriteByte ();
		break;

	default:
		Util_Printf ("i2c_transfer: bad call\n");
		result = I2C_NOK;
		break;
	}

	return (result);
}

